# Quick Reference: What to Copy from Local to Live

## TL;DR (Too Long; Didn't Read)

**Two things to sync:**

1. **Database** - Contains all admin settings (logo, banner, colors, company name, etc.)
2. **public/storage/** - Contains all uploaded images (logos, banners, products, etc.)

---

## Database Tables to Export

```bash
# Export these tables from local database
mysqldump -u root -p multivendor \
  business_settings \
  banners \
  brands \
  categories \
  shop_settings \
  settings \
  > settings_backup.sql

# Then import on live:
mysql -u live_user -p'password' live_db < settings_backup.sql
```

**Key Tables Explained:**

| Table | What It Contains | Example |
|-------|------------------|---------|
| `business_settings` | ALL admin configs | Company name, logo paths, colors, email, phone |
| `banners` | Homepage banners | Banner images and positions |
| `brands` | Brand info | Brand names and logos |
| `categories` | Product categories | Category names and icons |
| `shop_settings` | Shop configs | Per-vendor customizations |

---

## File Folders to Copy

```bash
# Copy this entire folder from local to live:
public/storage/

# Inside it:
├── company/          ← Logos, favicon, footer images
├── banner/           ← Homepage banners
├── brand/            ← Brand logos
├── category/         ← Category icons
├── product/          ← Product images
├── shop/             ← Vendor logos
├── profile/          ← User profile pictures
├── seller/           ← Seller images
└── delivery-man/     ← Delivery staff photos
```

**Copy with:**
```bash
# Via SCP (over SSH)
scp -r public/storage username@yourdomain.com:/home/username/public_html/public/

# Or create tar archive (faster for large files)
tar -czf public_storage.tar.gz public/storage/
scp public_storage.tar.gz username@yourdomain.com:/tmp/
# Then extract on live: tar -xzf /tmp/public_storage.tar.gz
```

---

## Critical vs Nice-to-Have

### CRITICAL (Site Won't Work Without These)
✅ `public/storage/company/` - Company logos  
✅ `business_settings` table - All configurations  
✅ Database migrations must run first

### HIGHLY RECOMMENDED (Site Works But Looks Bad)
⚠️ `public/storage/banner/` - Homepage banners  
⚠️ `public/storage/product/` - Product images  
⚠️ `public/storage/shop/` - Vendor logos

### NICE TO HAVE (Optional)
⭕ `public/storage/profile/` - User profile pictures  
⭕ `public/storage/brand/` - Brand logos  
⭕ `public/storage/category/` - Category icons

---

## What NOT to Copy

❌ Don't copy:
- `vendor/` folder (run `composer install` instead)
- `node_modules/` folder (run `npm install` instead)
- `storage/logs/` (let server create)
- `bootstrap/cache/` (auto-generated)
- `public/assets/` (in git)
- `resources/views/` (in git)

---

## Step-by-Step (Fastest Method)

### On Local Machine

```bash
# 1. Export database
mysqldump -u root -p multivendor business_settings banners brands categories > config.sql

# 2. Compress storage folder
tar -czf storage.tar.gz public/storage/

# 3. Upload both
scp config.sql storage.tar.gz username@yourdomain.com:/home/username/public_html/
```

### On Live Server (SSH)

```bash
cd /home/username/public_html

# 1. Import database
mysql -u live_user -p'password' live_db < config.sql

# 2. Extract files
tar -xzf storage.tar.gz

# 3. Fix permissions
chmod -R 755 public/storage/

# 4. Clear cache
php artisan cache:clear
php artisan config:cache
php artisan view:clear

# 5. Verify
curl https://yourdomain.com
# Should show homepage with logo and banners
```

---

## Verify It Worked

```bash
# Check 1: Database
mysql -u live_user -p'password' live_db \
  -e "SELECT COUNT(*) FROM business_settings;"
# Should show: > 0

# Check 2: Files
ls -la /home/username/public_html/public/storage/
# Should show folders: company, banner, product, etc.

# Check 3: Website
# Open https://yourdomain.com in browser
# Should see:
# ✓ Logo in header
# ✓ Banners on homepage
# ✓ Products with images
# ✓ Company name in title/footer

# Check 4: Logs (no errors)
tail -20 /home/username/public_html/storage/logs/laravel.log
# Should show normal requests, no "undefined" errors
```

---

## Size Estimates

```
Typical folder sizes:
company/     ~2MB
banner/      ~15MB
brand/       ~5MB
category/    ~5MB
product/     ~200MB+ (depends on # of products)
shop/        ~20MB
profile/     ~50MB
delivery/    ~10MB
─────────────────────
TOTAL:       ~300MB+

Database:    ~5-50MB (depends on # of products/orders)
```

---

## If Something Goes Wrong

### Logo Not Showing
```bash
# 1. Check files exist
ls /home/username/public_html/public/storage/company/

# 2. Fix permissions
chmod -R 755 /home/username/public_html/public/storage/

# 3. Check error log
tail -50 /home/username/public_html/storage/logs/laravel.log

# 4. Clear cache
php artisan cache:clear
php artisan config:cache
```

### Settings Not Applied (Wrong Colors/Company Name)
```bash
# 1. Verify database imported
mysql -u live_user -p'password' live_db \
  -e "SELECT * FROM business_settings WHERE key LIKE '%company%';"

# 2. Clear Laravel cache
php artisan cache:clear

# 3. Check AppServiceProvider can read it
php artisan tinker
> DB::table('business_settings')->first()
```

### "Permission Denied" Errors
```bash
chmod -R 775 /home/username/public_html/public/storage/
chown -R www-data:www-data /home/username/public_html/public/storage/
```

---

## All-in-One Command

```bash
# Complete sync (copy and paste this entire block):

# ON LOCAL MACHINE:
mysqldump -u root -p multivendor business_settings banners brands categories > /tmp/config.sql && \
tar -czf /tmp/storage.tar.gz public/storage/ && \
scp /tmp/config.sql /tmp/storage.tar.gz username@yourdomain.com:/tmp/ && \
echo "✓ Upload complete"

# ON LIVE SERVER (SSH):
cd /home/username/public_html && \
mysql -u live_user -p'password' live_db < /tmp/config.sql && \
cd /tmp && tar -xzf storage.tar.gz -C /home/username/public_html/ && \
chmod -R 755 /home/username/public_html/public/storage/ && \
cd /home/username/public_html && \
php artisan cache:clear && \
php artisan config:cache && \
php artisan view:clear && \
echo "✓ Sync complete! Check https://yourdomain.com"
```

---

## Summary Table

| What | Where | How | Critical? |
|-----|-------|-----|-----------|
| Settings | `business_settings` table | mysqldump | ✅ YES |
| Banners | `banners` table + `public/storage/banner/` | mysqldump + SCP | ✅ YES |
| Logos | `business_settings` + `public/storage/company/` | mysqldump + SCP | ✅ YES |
| Products | `products` table + `public/storage/product/` | mysqldump + SCP | ✅ YES |
| Brands | `brands` table + `public/storage/brand/` | mysqldump + SCP | ⚠️ MEDIUM |
| Categories | `categories` table + `public/storage/category/` | mysqldump + SCP | ⚠️ MEDIUM |
| Profiles | `business_settings` + `public/storage/profile/` | mysqldump + SCP | ⭕ LOW |

---

## Before You Start

- [ ] Backup live database: `mysqldump -u live_user -p live_db > backup_before_sync.sql`
- [ ] Verify .env is correct on live
- [ ] Check you have SSH/SCP access
- [ ] Ensure live database is empty or ready for sync
- [ ] Have ~500MB free space on live server

---

## After You Finish

- [ ] Visit https://yourdomain.com - should show logo and banners
- [ ] Check products display with images
- [ ] Check no errors in logs: `tail storage/logs/laravel.log`
- [ ] Test adding product to cart
- [ ] Test checkout process
- [ ] Keep backup of the sync files locally

---

**Questions?** See `LOCAL_TO_LIVE_SYNC_GUIDE.md` for detailed instructions.

